-- https://redmine.prodat-sql.de/issues/20431
-- eigenes Schema
CREATE SCHEMA IF NOT EXISTS tedifact;

-- ENUM-Table für unterstützte EDIFACT-Messagetypen
CREATE TABLE tedifact.message_types
(
    message_type varchar PRIMARY KEY
);


-- Importtabelle für eingehende EDIFACT-Nachrichten
CREATE TABLE tedifact.edi_incoming
(
    inc_id serial PRIMARY KEY,                         -- technische ID
    inc_message_type varchar NOT null
      REFERENCES tedifact.message_types,               -- identifiziert den EDIFACT-Nachrichtentyp
    inc_origin character varying(60) NOT null,         -- identifiziert die einzelne EDI-Message
    inc_vendor varchar NOT null,                       -- der Lieferant
    inc_edi_version_message varchar NOT null,          -- EDI-Version der Nachricht
    inc_edi_version_parsed varchar,                    -- EDI-Version, mit der die Nachricht geparst wurde
    inc_content varchar NOT NULL,                      -- der eigentliche EDI-Inhalt
    inc_insert timestamp without time zone NOT null,   -- Zeitpunkt des Eintreffens der EDI-Message
    inc_processing timestamp without time zone,        -- Zeitpunkt der letzten Verarbeitung der EDI-Message (erfolgreich oder nicht)
    inc_error_message text,                            -- Fehlermeldung der letzten Verarbeitung
    inc_success character varying(40)                  -- bei erfolgreicher Verarbeitung erzeugtes PRODAT-Artefakt
);

-- Pro Messagetyp muss die Nachrichten-ID eindeutig sein (unabhängig vom Versender?)
CREATE UNIQUE INDEX edi_incoming__message_type__inc_origin__uq
    ON tedifact.edi_incoming( inc_message_type, inc_origin) ;

-- Set von Trennungszeichen am Anfang jeder Nachricht
CREATE TYPE tedifact.service_string_advice AS (
  component_data_element_separator varchar(1),   -- Subsegmenttrenner, meist +
  data_element_separator varchar(1),             -- Datenelementtrenner, meist :   
  decimal_notation varchar(1),                   -- Dezimaltrenner, meist .
  release_indicator varchar(1),                  -- Escapecharakter, meist ?
  segment_terminator varchar(1)                  -- Segment- (oder Zeilen) -trenner, meist '
);

-- Tabelle für die Spezifikationen der Subsegmente (oder Datensatzcomponenten)
CREATE TABLE tedifact.subsegments (
  subs_id                serial PRIMARY KEY, -- Technische ID
  subs_release_type      varchar NOT null,   -- Releasetyp - EDIFACT oder CEFACT
  subs_release_version   varchar NOT null,   -- Releaseversion
  subs_segment           varchar NOT null,   -- zugehöriges Segment
  subs_pos               integer NOT null,   -- Subsegmentposition
  subs_code              varchar NOT null,   -- Code des Subsegments
  subs_bez               varchar NOT null,   -- Bezeichner des Subsegments
  subs_mandatory         boolean NOT null,   -- Subsegment obligatorisch oder optional?
  subs_repeat            integer,            -- Wie oft kommt das Subsegment vor?
  subs_insert_date       timestamp NOT null  -- Einfügezeitpunkt
);

-- An einer Position in einem Segment kann sich nur ein Subsegment befinden.
CREATE UNIQUE INDEX edifact_subsegments__subs_segment__subs_pos__uq
    ON tedifact.subsegments( subs_segment, subs_pos );
--

-- Insertdate automatisch hinzufügen
CREATE OR REPLACE FUNCTION subsegments__b_iu() RETURNS TRIGGER AS $$
BEGIN

  IF new.subs_insert_date IS null THEN
    new.subs_insert_date := now();
  END IF;

  RETURN new;

END $$ LANGUAGE plpgsql;

CREATE TRIGGER subsegments__b_iu
  BEFORE INSERT OR UPDATE
  ON tedifact.subsegments
  FOR EACH ROW
  EXECUTE PROCEDURE subsegments__b_iu();
--

-- Tabelle für die Spezifikationen der Datenelemente
CREATE TABLE tedifact.dataelements (
  data_id           serial PRIMARY KEY,                               -- Technische ID
  data_subs_id      integer NOT null REFERENCES tedifact.subsegments, -- Verweis auf das subsegement
  data_pos          integer NOT null,                                 -- Datenelementposition
  data_code         varchar NOT null,                                 -- Code des Datenelements
  data_bez          varchar NOT null,                                 -- Bezeichner des Datenelements
  data_mandatory    boolean NOT null,                                 -- Datenelement obligatorisch oder optional?
  data_alpha        boolean NOT null,                                 -- Kommen Buchstaben im Datenelement vor?
  data_numeric      boolean NOT null,                                 -- Kommen Ziffern im Datenelement vor?
  data_length       integer NOT null,                                 -- Länge des Datenelements
  data_length_exact boolean NOT null,                                 -- Maximal- oder exakte Länge?
  data_insert_date  timestamp NOT null                                -- Einfügezeitpunkt
);
--

-- Insertdate automatisch hinzufügen
CREATE OR REPLACE FUNCTION dataelements__b_iu() RETURNS TRIGGER AS $$
BEGIN

  IF new.data_insert_date IS null THEN
    new.data_insert_date := now();
  END IF;

  RETURN new;

END $$ LANGUAGE plpgsql;

CREATE TRIGGER dataelementss__b_iu
  BEFORE INSERT OR UPDATE
  ON tedifact.dataelements
  FOR EACH ROW
  EXECUTE PROCEDURE dataelements__b_iu();
--

-- Tabelle für die Spezifikationen der Segmente (oder Zeilen) innerhalb einer bestimmten Message
-- Ausschnitt einer Messagespezifikation
-- ├─UNH  ×1   (M)
-- ├─BGM  ×1   (M)
-- ├─DTM  ×35  (M)
-- ├─PAI  ×1   (C)
-- ├─ALI  ×5   (C)
-- ├─IMD  ×999   (C)
-- ├─FTX  ×99  (C)
-- ├─GIR  ×10  (C)
-- ├─Group 1  ×9999  (C)
-- │─├─RFF  ×1   (M)
-- │─└─DTM  ×5   (C)
-- ├─Group 2  ×1   (C)
-- │─├─AJT  ×1   (M)
-- │─└─FTX  ×5   (C)
-- ├─Group 3  ×99  (C)
-- │─├─NAD  ×1   (M)
-- │─├─LOC  ×25  (C)
-- │─├─FII  ×5   (C)
-- │─├─Group 4  ×99  (C)
-- │─│─├─RFF  ×1   (M)
-- │─│─└─DTM  ×5   (C)
CREATE TABLE tedifact.segments (
  segm_id serial PRIMARY KEY,                                              -- technische ID
  segm_release_version varchar NOT null,                                   -- verwendete Releaseversion
  segm_message_type varchar REFERENCES tedifact.message_types NOT null,    -- Messegtyp
  segm_content_segment boolean NOT null,                                   -- tatsächliches Segment oder virtueller Knoten?
  segm_group integer NOT null,                                             -- Gruppennummer des Segments
  segm_group_level integer,                                                -- Gruppenlevel (Root=1)
  segm_supergroup integer,                                                 -- Nummer der Obergruppe
  segm_segment varchar,                                                    -- Segment-ID (3 Großbuchstaben)
  segm_repeat integer NOT null,                                            -- Wie oft kann dieses Segment in der Nachricht vorkommen?
  segm_mandatory boolean NOT null,                                         -- Ist das Segment obligatorisch?
  segm_insert_date timestamp NOT null                                      -- Einfügedatum
);

-- Nur ein Segment pro Release, Messegetyp und Gruppe
CREATE UNIQUE INDEX IF NOT EXISTS edifact_segments__segm_group__segm_segment__uq
    ON tedifact.segments( segm_release_version, segm_message_type, segm_group, segm_segment );
--

-- Einfügedatum wird automatisch gesetzt
CREATE OR REPLACE FUNCTION segments__b_iu() RETURNS TRIGGER AS $$
BEGIN

  IF new.segm_insert_date IS null THEN
    new.segm_insert_date := now();
  END IF;

  RETURN new;

END $$ LANGUAGE plpgsql;

CREATE TRIGGER segments__b_iu
  BEFORE INSERT OR UPDATE
  ON tedifact.segments
  FOR EACH ROW
  EXECUTE PROCEDURE segments__b_iu();
--


-- Tabelle für geparste Segmentinformationen
CREATE TABLE tedifact.edi_parsed_segments
(
    psegm_id serial PRIMARY KEY,                                  -- technische ID
    psegm_inc_id integer NOT null                                 -- Verweis auf die Message
      REFERENCES tedifact.edi_incoming,
    psegm_segm_id integer NOT null REFERENCES tedifact.segments,  -- Referenz auf die zugehörige Segmentbeschreibung
    psegm_content varchar NOT null,                               -- Segmentinhalt
    psegm_insert_date timestamp without time zone NOT null,       -- Zeitpunkt des Eintreffens der EDI-Message
    psegm_success boolean                                         -- Parsen erfolgreich?
);
--

-- Insert-Datum wird automatisch gesetzt
CREATE OR REPLACE FUNCTION edi_parsed_segments__b_iu() RETURNS TRIGGER AS $$
BEGIN

  IF new.psegm_insert_date IS null THEN
    new.psegm_insert_date := now();
  END IF;

  RETURN new;

END $$ LANGUAGE plpgsql;
--

CREATE TRIGGER edi_incoming_segments__b_iu
  BEFORE INSERT OR UPDATE
  ON tedifact.edi_parsed_segments
  FOR EACH ROW
  EXECUTE PROCEDURE edi_parsed_segments__b_iu();
--

-- Tabelle für geparste Subsegmentinformationen
CREATE TABLE tedifact.edi_parsed_subsegments
(
    psubs_id serial PRIMARY KEY,                                   -- technische ID
    psubs_inc_id integer NOT null                                  -- Verweis auf die Message
      REFERENCES tedifact.edi_incoming,
    psubs_subs_id integer NOT null REFERENCES tedifact.subsegments,  -- Referenz auf die zugehörige Subsegmentbeschreibung
    psubs_psegm_id integer NOT null REFERENCES tedifact.edi_parsed_segments,
                                                                   -- Referenz auf das übergeordnete Segment
    psubs_content varchar NOT null,                                -- Subsegmentinhalt
    psubs_insert_date timestamp without time zone NOT null,        -- Zeitpunkt des Eintreffens der EDI-Message
    psubs_success boolean,                                         -- Parsen erfolgreich?
    psubs_error varchar                                            -- Fehler beim Parsen, sofern aufgetreten
);
--

-- Insert-Datum wird automatisch gesetzt
CREATE OR REPLACE FUNCTION edi_parsed_subsegments__b_iu() RETURNS TRIGGER AS $$
BEGIN

  IF new.psubs_insert_date IS null THEN
    new.psubs_insert_date := now();
  END IF;

  RETURN new;

END $$ LANGUAGE plpgsql;
--

CREATE TRIGGER edi_parsed_subsegments__b_iu
  BEFORE INSERT OR UPDATE
  ON tedifact.edi_parsed_subsegments
  FOR EACH ROW
  EXECUTE PROCEDURE edi_parsed_subsegments__b_iu();
--

-- Tabelle für die geparsten Daten
CREATE TABLE tedifact.edi_parsed_dataelements
(
    pdata_id serial PRIMARY KEY,                                   -- technische ID
    pdata_inc_id integer NOT null                                  -- Verweis auf die Message
      REFERENCES tedifact.edi_incoming,
    pdata_psubs_id integer NOT null                                -- Verweis auf das übergeordnete Subsegment
      REFERENCES tedifact.edi_parsed_subsegments,
    pdata_data_id integer NOT null REFERENCES tedifact.dataelements,  -- Referenz auf die zugehörige Beschreibung des Datenelements
    pdata_content varchar NOT null,                                -- Inhalt des Datenelements
    pdata_insert_date timestamp without time zone NOT null,             -- Zeitpunkt des Eintreffens der EDI-Message
    pdata_success boolean,                                         -- Parsen erfolgreich?
    pdata_error varchar                                            -- Fehler beim Parsen, sofern aufgetreten
);
--

-- Insert-Datum wird automatisch gesetzt
CREATE OR REPLACE FUNCTION edi_parsed_dataelements__b_iu() RETURNS TRIGGER AS $$
BEGIN

  IF new.pdata_insert_date IS null THEN
    new.pdata_insert_date := now();
  END IF;

  RETURN new;

END $$ LANGUAGE plpgsql;
--

CREATE TRIGGER edi_parsed_dataelements__b_iu
  BEFORE INSERT OR UPDATE
  ON tedifact.edi_parsed_dataelements
  FOR EACH ROW
  EXECUTE PROCEDURE edi_parsed_dataelements__b_iu();
--

-- Informationen über Sender bzw. empfänder von EDI-Nachrichten
CREATE TABLE tedifact.customers(
  cust_id serial PRIMARY KEY,                                           -- technische ID
  cust_customer_cimpcs varchar NOT null,                                -- PRODAT-Kunde
  cust_customer_external varchar NOT null,                              -- Lieferant bzw. Kunde des PRODAT-Kunden (Selbstbezeichnung laut EDI)
  cust_a2_krz varchar NOT null,                                         -- Lieferant bzw. Kunde des PRODAT-Kunden (adk2-Tabelle)
  cust_message_type varchar NOT null REFERENCES tedifact.message_types, -- Nachrichtentyp
  cust_message_incoming boolean NOT null DEFAULT false,                 -- eingehende Nachricht?
  cust_message_outgoing boolean NOT null DEFAULT false                  -- ausgehende Nachricht?
);
--

-- Es kann nur einen Kundenkunden geben.
CREATE UNIQUE INDEX IF NOT EXISTS edifact_customers__cimpcs_external__uq
    ON tedifact.customers( cust_customer_cimpcs, cust_customer_external );


-- Tabelle für die Übersetzung eines EDI-Datensatzes in einen PRODAT-Datensatz
-- Der Datensatz wird beim Parsen nur berücksichtigt, wenn entweder keine Bedingung angegeben ist, oder diese Bedingung erfüllt ist.
CREATE TABLE tedifact.edi_translation(
  trans_id serial PRIMARY KEY,                                   -- technische ID
  trans_customer integer NOT null REFERENCES tedifact.customers, -- Verweis auf kundenspezifischen Nachrichtentyp
  trans_data_code varchar NOT null,                              -- Code des EDI-Datenelements
  trans_segment varchar NOT null,                                -- Segment des EDI-Datenelement
  trans_subs_pos integer NOT null,                               -- Position des EDI-Subsegments innerhalb des Segments
  trans_condition_data_code varchar,                             -- Code des EDI-Datenelements für die Bedingung
  trans_condition_subs_pos integer,                              -- Position des EDI-Subsegments für die Bedingung innerhalb des Segments
  trans_condition_value varchar,                                 -- Wert des EDI-Datenelements für die Bedingung
  trans_prodat_meaning varchar NOT null                          -- Text für die Bedeutung der Daten in PRODAT
);
--

-- Pro kundenspezifischen Nachrichtentyp kann es nur eine Übersetzung geben.
CREATE UNIQUE INDEX IF NOT EXISTS edifact_translation__customer__meaning__uq
    ON tedifact.edi_translation( trans_customer, trans_prodat_meaning );
--

-- Hilfstyp für das Parsen von EDI-Botschaften
-- siehe tedifact.dataelement__read
-- siehe tedifact.ldsdok__ordrsp__get
CREATE TYPE tedifact.edi_parsed_dataelements_short AS (
  pdata_id integer,
  pdata_content varchar,
  meaning varchar
);
--
